#lang racket/gui

; --- imports ---
(require framework) ; keymap:setup-editor, color:text%
(require parser-tools/lex) ; for lexer-src-pos

; --- exports ---
(provide ide-editor%)

; --- body ---
(define ide-editor%
  (class color:text%
    (super-new)
    (init frame
          tab)
    
    ; - fields -
    (define parent-frame frame)
    (define parent-tab tab)

    ; - methods -
    (define/public (saved-or-empty?)
      (let ([filename (send this get-filename)])
        (or (and (not filename) (eq? (send this get-text) "")) ; new file, empty content
            (and filename (not (send this is-modified?)))))) ; existing file, content not modified

    (define/augment (can-close?) ; Can we close this editor?
      (or (saved-or-empty?) ; editor of the tab is empty or saved -> the tab can be closed right away
          ; otherwise (ie. not empty and not saved), ask what to do
          (let ([result (message-box/custom "Warning" "There are unsaved modifications inside the buffer.\nClose anyway?" "Save and close" "Cancel" "Close without saving" parent-frame '(default=2 caution) 2)])
            (cond [(= result 1) (send this save-file #f 'text)]
                  [(= result 2) #f]
                  [(= result 3) #t]))))

    ; - lexer -
    ; input: the symbol returned by the lexer, output: the style name as a string
    ; style-names should match those set in the editor style-list%
    (define (token-sym->style sym)
      (cond [(eq? 'directive sym) "directive"] ; directive style
            [(eq? 'instruction sym) "instruction"] ; instruction style
            [(eq? 'register sym) "register"] ; register name style
            [(eq? 'numeric sym) "numeric"] ; numeric style
            [(eq? 'label sym) "label"] ; label style
            [(eq? 'comment sym) "comment"] ; comment style
            ; ...
            [(eq? 'uncoloured sym) "base-font"])) ; base-font, no colouring

    ; https://github.com/racket/parser-tools/blob/master/parser-tools-lib/parser-tools/examples/calc.rkt
    ; /!\ We need to exhaust all patterns: as soon as an unmatched character appears (e.g. input by the user), an exception is raised.
    (define (get-token port offset mode) ; lexer
      ; We use the 3-argument version (ie. with offset and mode) so that a backup distance can be set to backtrack when colouring.
      ; Without such backup distance, when typing "add", the lexer is called on "a" then "d", then "d" thus matching 3 chars separately and not "add"
      (define-tokens valued-tokens (directive instruction register numeric label comment)) ; if too slow, move to define-empty-tokens since lexeme unneeded it seems
      (define-empty-tokens empty-tokens (eof no-color uncoloured))
      (let* ([result ; remember that the lexer always favours longest matches
              ((lexer-src-pos ; 'eof and 'no-color are special symbols, no need for such styles
                [(eof) (token-eof)] ; required

                [(concatenation (union ".equ" ".equiv" ".org" ".byte" ".ascii" ".string" ".include" ".text" ".section") whitespace) (token-directive lexeme)] ; assembler directives
                [(union "hi8" "lo8") (token-directive lexeme)]

                [(concatenation
                  whitespace ; this disables colouring of the first opcode if at beginning of file (ie. with no whitespace before it)
                  (union
                   ; arithmetic & logic
                   "add" "adc" "adiw" "sub" "subi" "sbc" "sbci" "sbiw"
                   "and" "andi" "or" "ori" "eor" "com" "neg"
                   "sbr" "cbr" "inc" "dec" "tst" "clr" "ser"
                   "mul" "muls" "mulsu" "fmul" "fmuls" "fmulsu" "des"
                   ; flow control
                   "rjmp" "ijmp" "eijmp" "jmp" "rcall" "icall" "eicall" "call" "ret" "reti"
                   "cpse" "cp" "cpc" "cpi" "sbrc" "sbrs" "sbic" "sbis"
                   "brbs" "brbc" "breq" "brne" "brcs" "brcc" "brsh" "brlo" "brmi" "brpl"
                   "brge" "brlt" "brhs" "brhc" "brts" "brtc" "brvs" "brvc" "brie" "brid"
                   ; data transfer
                   "mov" "movw" "ldi" "lds" "ld" "ldd" "sts" "st" "std" "lpm" "elpm" "spm"
                   "in" "out" "push" "pop" "xch" "las" "lac" "lat"
                   ; bit & bit test
                   "lsl" "lsr" "rol" "ror" "asr" "swap" "sbi" "cbi" "bst" "bld" "bset" "bclr"
                   "sec" "clc" "sen" "cln" "sez" "clz" "sei" "cli" "ses" "cls" "sev" "clv" "set" "clt" "seh" "clh"
                   ; MCU control
                   "break" "nop" "sleep" "wdr") whitespace) (token-instruction lexeme)] ; it seems unneeded to keep the token value (lexeme)
                [(concatenation
                  ;whitespace
                  #\r (repetition 1 2 numeric)) (token-register lexeme)] ; register names
                [(concatenation (repetition 1 +inf.0
                                            (intersection (complement #\;) graphic) ;graphic ;(intersection (complement punctuation) graphic)
                                            ) #\:) (token-label lexeme)] ; label

                ; string (even if no colour, this is to avoid colouring inside a string!)
                [(concatenation #\" (complement (concatenation any-string #\" any-string)) #\") (token-uncoloured)] ; escaping inside string \" not supported yet
                
                [(concatenation
                  ;whitespace ; this breaks colouring in label (1:), arith expr, and first numerical values if at beginning of file (ie. with no whitespace before it)
                  (union
                   (repetition 1 +inf.0 numeric) ; 1234
                   (concatenation "0x" (repetition 1 +inf.0 (union (char-range #\A #\F) (char-range #\a #\f) numeric))) ; 0x0A1F
                   (concatenation "0b" (repetition 1 +inf.0 (union #\0 #\1))) ; 0b0010
                   )) (token-numeric lexeme)] ; numerical values
                ;[numeric (token-numeric lexeme)] ; digits
                ;[(concatenation "0x" (repetition 1 +inf.0 (union (char-range #\A #\F) (char-range #\a #\f) numeric))) (token-numeric lexeme)]
                ;[(concatenation "0b" (repetition 1 +inf.0 (union #\0 #\1))) (token-numeric lexeme)]
                
                [(concatenation #\; (repetition 0 +inf.0 (char-complement #\newline))) (token-comment lexeme)] ; (union "" #\newline) ; line comment
                [(concatenation "/*" (complement (concatenation any-string "*/" any-string)) "*/") (token-comment lexeme)] ; block comment
                
                ; this last pattern needs to be last
                [any-char (token-uncoloured)];(token-no-color)] ; collides with other patterns but since appearing last, it has a lower priority
                ; in the above pattern, the special token 'no-color is not suitable since keeping the previously set style (eg. color)
                ; hence, we introduce our own 'uncoloured token to explicitly revert to the base-font style (ie. no colour)
                )
               port)] ; returns (make-position-token action-result start-pos end-pos)
             [token (position-token-token result)]
             [start-pos-offset (position-offset (position-token-start-pos result))]
             [end-pos-offset (position-offset (position-token-end-pos result))])
        ; return the required 5 values
        ;(values (token-value token) (token-name token) #f start-pos-offset end-pos-offset)
        ; return the required 7 values (backup distance of 1 is enough)
        (values (token-value token) (token-name token) #f start-pos-offset end-pos-offset 1 #f)))

    ; - colouration events -
    (define/augment (on-edit-sequence) ; called once at the beginning of the edit sequence
      (send this reset-region 0 'end)) ; re-tokenise (recolour) everything at the start of edit sequence (needed to recolor on character delete)    
    (define/augment (after-edit-sequence)
      (send this reset-region 0 'end)) ; re-tokenise (recolour) everything after edit sequence

    ; - editor content modification events -
    (define/augment (after-insert start len)
      ;(send this reset-region start (+ start len)) ; re-tokenise (recolour); does not work
      (when (send this is-modified?) (send parent-tab set-marker-when-unset))) ; remove the condition check (is-modified?) if too slow
    (define/augment (after-delete start len)
      ;(send this reset-region start (+ start len)) ; re-tokenise (recolour); does not work
      (when (send this is-modified?) (send parent-tab set-marker-when-unset))) ; remove the condition check (is-modified?) if too slow

    ; - editor content modification & *caret movement* events -
    (define/override (on-default-char event) ; called upon caret movement and character insertion/deletion
      (super on-default-char event)
      (update-line-number))
    (define/override (on-default-event event) ; called upon a mouse event
      (super on-default-event event)
      (update-line-number))
    (define (update-line-number)
      (let ([line-number (add1 (send this position-line (send this get-end-position)))]) ; get-position / get-start-position also possible
        (send parent-frame update-line-label line-number))) ; find-line / position-location yet another possibility
    
    ; - load/save events -
    (define/augment (after-load-file success?)
      (when success?
        (let ([path (send this get-filename)])
          (let-values ([(base name must-be-dir?) (split-path path)])
            (send parent-tab set-title-and-reset-marker (path->string name)) ; set label which shows the filename
            ;(send parent-frame set-label (string-append "A-IDE - " (path->string path))) ; update window title
            (send parent-frame add-recent-entry (path->string name) (path->string path)) ; update recent entries
            ))))
    (define/augment (after-save-file success?)
      (after-load-file success?))
    
    ; - editor setup (style, set-style-list, etc.) -
    (send this set-max-undo-history 'forever) ; enable undo/redo
    (send this set-tabs '() 4 #f) ; 4 spaces per tab
    (send this set-paste-text-only #t) ; allow pasting only text (since pasting content from DrRacket can crash our simple lexer)
    ;(send this set-autowrap-bitmap #f) ; uncomment this line to disable showing little wrap arrows upon line wrapping

    ; Colouration setup
    ; add new styles into the list of the styles of the editor
    (let* ([style-list (send this get-style-list)]
           [basic-style (send style-list basic-style)]
           [font-delta (make-object style-delta%)]
           [font-style ; /!\ dynamic font adjustment does not work: restart needed /!\
            (let ([font-face (send parent-frame get-font-face)])
              (if (non-empty-string? font-face) ; when empty string:
                  (send font-delta set-face font-face)
                  (begin (send font-delta set-face #f) ; this way the font family is used when font face unavailable
                         (send font-delta set-family 'modern)))
              ; font size
              (send font-delta set-delta 'change-size (send parent-frame get-font-size))
              ; font colour for normal text (ie. text uncoloured by lexer)
              (send font-delta set-delta-foreground (send parent-frame get-edit-foreground))
              
              (send style-list find-or-create-style basic-style font-delta))] ; add new style in the list and return the new style
           
           [directive-delta (make-object style-delta%)]
           [directive-style (begin (send directive-delta set-delta-foreground "DarkCyan")
                                   (send style-list find-or-create-style font-style directive-delta))]
           [instruction-delta (make-object style-delta% 'change-bold)]
           [instruction-style (begin (send instruction-delta set-delta-foreground "RoyalBlue")
                                     (send style-list find-or-create-style font-style instruction-delta))]
           [register-delta (make-object style-delta%)]
           [register-style (begin (send register-delta set-delta-foreground "Olive")
                                  (send style-list find-or-create-style font-style register-delta))]
           [numeric-delta (make-object style-delta%)]
           [numeric-style (begin (send numeric-delta set-delta-foreground "ForestGreen")
                                 (send style-list find-or-create-style font-style numeric-delta))]
           [label-delta (make-object style-delta%)]
           [label-style (begin (send label-delta set-delta-foreground "MediumVioletRed")
                               (send label-delta set-delta-background "LavenderBlush")
                               (send style-list find-or-create-style font-style label-delta))]
           [comment-delta (make-object style-delta%)]
           [comment-style (begin (send comment-delta set-delta-foreground "Chocolate")
                                 (send style-list find-or-create-style font-style comment-delta))])
      (send this change-style font-style)
      ; give a name to styles for later reuse
      (send style-list new-named-style "base-font" font-style)
      (send style-list new-named-style "directive" directive-style)
      (send style-list new-named-style "instruction" instruction-style)
      (send style-list new-named-style "register" register-style)
      (send style-list new-named-style "numeric" numeric-style)
      (send style-list new-named-style "label" label-style)
      (send style-list new-named-style "comment" comment-style))
    ; start colorer (this method must be called once, eg. not on each after-edit-sequence)
    (send this start-colorer token-sym->style get-token '())
    
    ; Keymap setting
    ; BECAUSE this class extends color:text%, the keymap is already setup.
    ; The bindings are listed in the documentation at 'keymap:setup-global'.
    ;(define common-keymap (new keymap%))
    ;(keymap:setup-editor common-keymap)
    ;(send (send this get-keymap) chain-to-keymap common-keymap #f)

    ; Add new shortcuts: Ctrl-tab & Ctrl-Shift-tab, to switch tabs, etc.
    (let ([current-keymap (send this get-keymap)])
      (send current-keymap add-function "previous-tab" (lambda (in event) (send parent-frame select-next-tab -1)))
      (send current-keymap map-function "c:s:tab" "previous-tab")
      (send current-keymap add-function "next-tab" (lambda (in event) (send parent-frame select-next-tab +1)))
      (send current-keymap map-function "c:tab" "next-tab")
      (send current-keymap add-function "toggle-overwrite" (lambda (in event) (send in set-overwrite-mode (not (send in get-overwrite-mode)))))
      (send current-keymap map-function "c:m:o" "toggle-overwrite") ; "insert" shortcut key does not work
      
      (send current-keymap add-function "new-line-indent" ; automatically replicate indentation of the previous line upon new line insertion
            (lambda (in event) ; get the indent amount of the previous line
              (let* ([current-position (send in get-start-position)]
                     [current-paragraph-number (send in position-paragraph current-position)]
                     [current-paragraph-start (send in paragraph-start-position current-paragraph-number)]
                     ;[current-paragraph-end (send in paragraph-end-position current-paragraph-number)]
                     ;[current-paragraph-text (send in get-text current-paragraph-start current-paragraph-end)]
                     ;[indent-characters (takef (string->list current-paragraph-text) char-blank?)] ; the indent of the current paragraph (tab, spaces, etc.)
                     ; Plutôt reproduire les blancs qui sont *à gauche* de current-position.
                     [left-of-current-position (send in get-text current-paragraph-start current-position)]
                     [indent-characters (takef (string->list left-of-current-position) char-blank?)]) ; the indent of the current paragraph (tab, spaces, etc.)
                (send in insert (list->string (cons #\newline indent-characters)) current-position) ; insert NEWLINE & blanks
                )))
      (send current-keymap map-function "return" "new-line-indent")

      ; Shift-Tab: delete TAB on the left of the current position, if any
      (send current-keymap add-function "deindent"
            (lambda (in event)
              (let* ([current-position (send in get-start-position)]
                     [left-character (send in get-text (sub1 current-position) current-position)])
                (when (char=? (string-ref left-character 0) #\tab)
                  (send in delete (sub1 current-position) current-position)))))
      (send current-keymap map-function "s:tab" "deindent")
      
      )

    ))
