#lang racket/gui

; --- imports ---
;(require "ide-editor.rkt")
(require "ide-tab.rkt")
(require "constants.rkt")

; --- exports ---
(provide ide-tab-panel%)

; --- body ---
(define ide-tab-panel%
  (class tab-panel%
    (init frame)
    (super-new [choices (list default-tab-title)]
               [style '(flat-portable no-border can-reorder can-close new-button)]
               [callback (lambda (tab-panel event) (synchronise-children-tabs))])
    ; - fields -
    (define parent-frame frame)
    (define ids-children ; a list of pairs (tab-id . children)
      (list (cons 0 (new tab% [frame parent-frame] [tab-panel this] [tab-id 0]))))
    
    ; - override & augments -
    (define/override (on-new-request)
      (let* ([new-tab-id (length ids-children)] ; == (send this get-number), the number of tabs, obviously
             [child (new tab% [frame parent-frame] [tab-panel this] [tab-id new-tab-id])]
             [child-editor (send child get-editor)])
        (update-ids-children (append ids-children (list (cons new-tab-id child)))) ; 1. Update ids-children
        (send this append (send child get-title)) ;(send this append "(new buffer)") ; 2. Add tab
        (send parent-frame enable-menu-close-item #t)
        (send this set-selection new-tab-id) ; 3. Select tab
        (synchronise-children-tabs)
        ; 4. Editor settings
        (when (send parent-frame linewrap-item-checked?) (send child-editor auto-wrap #t))
        child-editor)) ; 5. Return the corresponding editor

    (define/augment (on-reorder former-ids)
      (update-ids-children ; Update ids-children
       (sort (map (lambda (new-id id-child) (cons new-id (cdr id-child))) former-ids ids-children) < #:key car))
      (synchronise-children-tabs)) ; update children because reordering keeps the selected id, but a new tab is assigned to that id...
    
    (define/override (on-close-request tab-id)
      (if (> (send this get-number) 1) ; only allow closing when there are at least two tabs
          (let ([tab-editor (get-editor tab-id)]) ; the editor of the tab that is to be closed (not always equal to current-editor!)
            (when (send tab-editor can-close?)
              (send this delete tab-id) ; 1. Remove tab
              (when (= (send this get-number) 1) (send parent-frame enable-menu-close-item #f)) ; disable Close menu item when only one tab remains
              ; 2. Update ids-children
              (let-values ([(before-tabs other-tabs) (split-at ids-children tab-id)])
                (let* ([after-tabs (cdr other-tabs)] ; other-tabs includes the removed tab
                       [tab-id-after-removal (if (null? after-tabs) (sub1 tab-id) tab-id)]) ; current tab after removal
                  (update-ids-children ; -1 to all elements of ids-children that are after the removed tab (ie. after-tabs)
                   (append before-tabs (map (lambda (id-child) (cons (sub1 (car id-child)) (cdr id-child))) after-tabs)))
                  ; 3. Activate another tab
                  (send this set-selection tab-id-after-removal) ; select tab after removal
                  (synchronise-children-tabs))))) ; update children because only selecting a tab does not trigger callback
          (bell))) ; somehow notify the user that the last tab cannot be closed
    
    ; - methods -
    (define (update-ids-children new-ids-children)
      (set! ids-children new-ids-children)
      (for ([id-child ids-children]) ; update tab-id of each tab% instance
        (let ([id (car id-child)] [child (cdr id-child)])
          (send child set-id id))))
    
    (define (synchronise-children-tabs) ; show the child corresponding to the currently selected tab
      (let ([current-child (get-current-child)])
        (send this change-children (lambda (children) (list current-child)))
        (send current-child focus))) ; give focus to the editor

    (define (get-child tab-id) (cdr (assq tab-id ids-children))) ; editor-canvas%
    (define (get-editor tab-id) (send (get-child tab-id) get-editor))
    (define (get-current-child) (get-child (send this get-selection))) ; editor-canvas%
    (define/public (get-current-editor) (send (get-current-child) get-editor))
    (define/public (get-all-editors) (map (lambda (id-child) (send (cdr id-child) get-editor)) ids-children))
    
    (define/public (select-next-tab [offset 1])
      (let* ([current-tab-id (send this get-selection)]
             [number-of-tabs (length ids-children)] ; of course, this is equal to (send this get-number)
             [next-tab-id (modulo (+ current-tab-id offset) number-of-tabs)])
        (send this set-selection next-tab-id)
        (synchronise-children-tabs))) ; update children because only selecting a tab does not trigger callback

    ; - initial actions -
    (send this set-selection 0) ; to select the initial tab (tab change callback not called)
    (synchronise-children-tabs) ; update children because only selecting a tab does not trigger callback
    (when (send parent-frame linewrap-item-checked?) (send (get-current-editor) auto-wrap #t))
    ))
