#lang racket/gui

; --- imports ---
(require framework) ; panel:vertical-dragable%
(require "ide-tab-panel.rkt")
(require "line-message.rkt")
(require "preferences-dialog.rkt")
(require "view-dialog.rkt")
(require "about-dialog.rkt")
(require "constants.rkt")

; --- exports ---
(provide ide-frame%)

; --- body ---
(define ide-frame%
  (class frame%
    (super-new [label "A-IDE"] ; then, tentatively load preferences from file
               [x (get-preference 'frame-x (lambda () frame-default-x) 'timestamp preferences-filename)]
               [y (get-preference 'frame-y (lambda () frame-default-y) 'timestamp preferences-filename)]
               [width (get-preference 'frame-width (lambda () frame-default-width) 'timestamp preferences-filename)]
               [height (get-preference 'frame-height (lambda () frame-default-height) 'timestamp preferences-filename)])

    ; - some fields -
    (define percentages (get-preference 'frame-percentages (lambda () frame-default-percentages) 'timestamp preferences-filename))
    (define recent-entries
      (let ([entries (get-preference 'recent-entries (lambda () '()) 'timestamp preferences-filename)]) ; recently opened files
        (filter (lambda (entry) (file-exists? (cdr entry))) entries)))
    
    ; - overrides & augments -
    (define/augment (can-close?)
      (if (andmap (lambda (editor) (send editor saved-or-empty?)) (send tab-panel get-all-editors)) ; check whether all editors are ready to close
          #t ; (content saved) or (no content in new file), we can close
          ; "Unsaved modifications. Close anyway?"
          (let ([result (message-box/custom "Warning" "There are unsaved modifications inside at least one buffer.\nClose anyway?" "Close all buffers without saving" "Cancel" #f this '(default=2 caution) 2)])
            (cond [(= result 1)
                   ; Save all editors. /!\ if editors have no filename yet, many filenames would be asked...
                   ;(map (lambda (editor) (send editor save-file #f 'text)) (send tab-panel get-all-editors))
                   #t] ; Instead, we just ask to close all without saving or cancel.
                  [(= result 2) #f]))))

    (define/augment (on-close)
      ; kill thread-id if still running (no need to thread-wait, just kill)
      (when (and thread-id (not (thread-dead? thread-id))) ; check if valid thread before thread-dead?
        (kill-thread thread-id)) ; no need to wait, we kill the thread
      ; save preferences
      (if (send reset-item is-checked?)
          ; write default preferences
          (put-preferences '(frame-x frame-y frame-width frame-height frame-percentages output-shown font-face font-size edit-foreground edit-background output-foreground output-background)
                           (list frame-default-x frame-default-y
                                 frame-default-width frame-default-height
                                 frame-default-percentages
                                 default-output-shown
                                 default-font-face default-font-size
                                 default-edit-foreground default-edit-background default-output-foreground default-output-background)
                           #f preferences-filename)
          ; otherwise
          (let ([output-shown? (= (length (send v0-layout get-children)) 2)])
            (put-preferences '(frame-x frame-y frame-width frame-height frame-percentages output-shown font-face font-size edit-foreground edit-background output-foreground output-background)
                             (list (send this get-x) (send this get-y)
                                   (send this get-width) (send this get-height)
                                   (if output-shown?
                                       (send v0-layout get-percentages) ; get the current percentages
                                       percentages) ; get the percentages that were used before output was hidden
                                   output-shown?
                                   (send view-dialog get-font-face) (send view-dialog get-font-size)
                                   (let ([c (send view-dialog get-edit-foreground)]) (list (send c red) (send c green) (send c blue)))
                                   (let ([c (send view-dialog get-edit-background)]) (list (send c red) (send c green) (send c blue)))
                                   (let ([c (send view-dialog get-output-foreground)]) (list (send c red) (send c green) (send c blue)))
                                   (let ([c (send view-dialog get-output-background)]) (list (send c red) (send c green) (send c blue))))
                             #f preferences-filename)))
      ; append recent entries, auto-wrap, etc. as other preferences
      (put-preferences '(recent-entries linewrap avr-as avr-ld avr-objcopy avr-size avrdude avrdude-conf avr-as-isa-mcu avrdude-part avrdude-programmer avr-port)
                       (list recent-entries
                             (send linewrap-item is-checked?)
                             (send pref-dialog get-avr-as)
                             (send pref-dialog get-avr-ld)
                             (send pref-dialog get-avr-objcopy)
                             (send pref-dialog get-avr-size)
                             (send pref-dialog get-avrdude)
                             (send pref-dialog get-avrdude-conf)
                             (send pref-dialog get-avr-as-isa-mcu)
                             (send pref-dialog get-avrdude-part)
                             (send pref-dialog get-avrdude-programmer)
                             (send pref-dialog get-port))
                       #f preferences-filename))

    (define/override (on-drop-file path)
      (let ([current-editor (send tab-panel get-current-editor)])
        ; if the editor of the current tab can be closed (saved-or-empty?), reuse the tab
        (if (send current-editor saved-or-empty?)
            ; reuse the current tab's editor
            (send current-editor load-file path 'text) ; tab label to be set in the 'on-load...' callback of ide-editor%
            ; otherwise, new tab
            (let ([new-editor (send tab-panel on-new-request)]) ; creates a new tab and returns the corresponding editor
              (send new-editor load-file path 'text)
              ; NB. if when a new tab is created it is not shown, we need to set-selection and call (children-update)
              ))))

    ; - methods -
    (define/public (enable-menu-close-item v) (send menu-close-item enable v)) ; enable or disable the Close menu item
    (define/public (linewrap-item-checked?) (send linewrap-item is-checked?))
    (define/public (update-line-label line-number) (send line-label set-when-needed line-number))
    (define/public (select-next-tab offset) (send tab-panel select-next-tab offset))
    (define/public (get-font-face) (send view-dialog get-font-face))
    (define/public (get-font-size) (send view-dialog get-font-size))

    (define/public (get-edit-foreground) (send view-dialog get-edit-foreground))
    (define/public (get-edit-background) (send view-dialog get-edit-background))
    
    (define (populate-recent-menu)
      (for-each (lambda (item) (send item delete)) (send recent-menu get-items)) ; first erase menu
      (if (null? recent-entries) ; second populate
          (send recent-menu enable #f) ; if no entry, disable menu
          (begin ; otherwise, populate and enable menu
            (for-each (lambda (entry) (let ([filename (car entry)] [path (cdr entry)])
                                        (new menu-item% [label filename] [parent recent-menu] [callback (lambda (menu-item event) (on-drop-file path))])))
                      recent-entries)
            (send recent-menu enable #t))))
    
    (define/public (add-recent-entry filename path)
      (let ([entry (cons filename path)])
        (when (not (member entry recent-entries))
          (set! recent-entries (cons entry (if (> (length recent-entries) 4) (take recent-entries 4) recent-entries)))
          (populate-recent-menu))))
    
    (define (callback-find str dir case?)
      (let* ([current-editor (send tab-panel get-current-editor)]
             [position (if (non-empty-string? str)
                           (begin ; we have to first unselect otherwise the same entry is repeatedly found on "Find next"...
                             (if (eq? dir 'forward) 
                                 (send current-editor set-position (send current-editor get-end-position))
                                 (send current-editor set-position (send current-editor get-start-position)))
                             (send current-editor find-string str dir 'start 'eof #t case?)) #f)])
        (when position
          (if (eq? dir 'forward)
              (send current-editor set-position position (+ position (string-length str)))
              (send current-editor set-position (- position (string-length str)) position)))))

    (define (callback-replace str-from str-to dir case?) ; string-replace also available
      (let* ([current-editor (send tab-panel get-current-editor)]
             [position (if (non-empty-string? str-from)
                           (begin ; we have to first unselect otherwise the same entry is repeatedly replaced on "Replace next"...
                             (if (eq? dir 'forward) 
                                 (send current-editor set-position (send current-editor get-end-position))
                                 (send current-editor set-position (send current-editor get-start-position)))
                             (send current-editor find-string str-from dir 'start 'eof #t case?)) #f)])
        (when position
          (if (eq? dir 'forward)
              (begin
                (send current-editor delete position (+ position (string-length str-from)))
                (send current-editor insert str-to position)
                (send current-editor set-position position (+ position (string-length str-to))))
              (let ([left-pos (- position (string-length str-from))])
                (send current-editor delete left-pos position)
                (send current-editor insert str-to left-pos)
                (send current-editor set-position left-pos (+ left-pos (string-length str-to))))))))

    (define (callback-toggle)
      (if (= (length (send v0-layout get-children)) 2)
          (begin ; remove child
            (set! percentages (send v0-layout get-percentages)) ; for restoring the panel
            (send v0-layout delete-child output-canvas))
          (begin ; add child
            (send v0-layout add-child output-canvas) ; appends at the end
            (send v0-layout set-percentages percentages))))

    (define (callback-explorer)
      (let* ([current-editor (send tab-panel get-current-editor)] [pathname (send current-editor get-filename)])
        (when pathname
          (let ([folder (some-system-path->string (path-only pathname))] [os (system-type)])
            (when folder
              (let*-values ([(command args) (cond [(eq? os 'unix) (values "xdg-open" (list folder))]
                                                  [(eq? os 'macosx) (values "open" (list folder))] ; untested
                                                  [(eq? os 'windows) (values "cmd.exe" (list "/c" "start" "\"\"" folder))])]
                            [(process-id port-stdout port-stdin port-stderr) (apply subprocess #f #f #f (find-executable-path command) args)])
                (close-input-port port-stdout)
                (close-output-port port-stdin)
                (close-input-port port-stderr)))))))
    
    (define (callback-homepage) ; https://stackoverflow.com/questions/38147620/shell-script-to-open-a-url
      (let ([url "https://www.sci.kanagawa-u.ac.jp/info/abossard/a-ide/"] [os (system-type)])
        (let*-values ([(command args) (cond [(eq? os 'unix) (values "xdg-open" (list url))]
                                            [(eq? os 'macosx) (values "open" (list url))] ; untested
                                            [(eq? os 'windows) (values "cmd.exe" (list "/c" "start" url))])]
                      [(process-id port-stdout port-stdin port-stderr) (apply subprocess #f #f #f (find-executable-path command) args)])
          (close-input-port port-stdout)
          (close-output-port port-stdin)
          (close-input-port port-stderr))))
    
    (define/public (callback-about) (send about-dialog show #t)) ; public for 'application-about-handler' in main.rkt

    (define thread-id #f) ; #f is the initial value. Once a thread is created, thread-id is always a thread descriptor
    (define (callback-build upload-after-build?)
      ; if file does not exist, ask to save, otherwise if modifications not saved, ask whether to continue as is, save and build or cancel.
      (let* ([current-editor (send tab-panel get-current-editor)]
             [pathname (send current-editor get-filename)] ; we have to call get-filename again as possibly modified since last time
             [do-build (if (not pathname)
                           ; new buffer, either (save and build) or (cancel)
                           (let ([result (message-box/custom "Warning" "Building requires saving to a file. Continue?" "Save and build" "Cancel" #f this '(default=1 caution) 2)])
                             (cond [(= result 1) (send current-editor save-file #f 'text)]
                                   [(= result 2) #f]))
                           ; otherwise, file exists, check if there are unsaved modifications
                           (if (send current-editor is-modified?)
                               (let ([result (message-box/custom "Warning" "There are unsaved modifications. Build anyway?" "Save and build" "Cancel" "Build without saving" this '(default=1 caution) 2)])
                                 (cond [(= result 1) (send current-editor save-file #f 'text)]
                                       [(= result 2) #f]
                                       [(= result 3) #t]))
                               ; otherwise (there are no unsaved modifications)
                               #t))])
        ; unless cancel was selected, go on
        (when (and do-build (or (not thread-id) (thread-dead? thread-id)))
          (set! thread-id
                (thread (lambda () ; (thread ...) directly returns without waiting for completion, of course
                          (let ([to-disable (list build-button upload-button build-item upload-item)])
                            (for-each (lambda (control) (send control enable #f)) to-disable)
                            (begin-busy-cursor)
                            (send build-message set-label "building...")
                            (send output-editor lock #f)
                            (send output-editor erase) ; /!\ removes styling
                            (send output-editor change-style output-editor-style) ; so, we re-apply the style here
                            (send output-editor insert "--- building has started ---\n")
                            ; Now that the editor has been locked, we can proceed
                            (let* ([source-file-name (some-system-path->string (file-name-from-path pathname))]
                                   [path-s pathname] ; path to code source file
                                   [path-o (path-replace-extension pathname ".o")] ; path to .o file
                                   [path-elf (path-replace-extension pathname ".elf")] ; path to .elf file
                                   [path-hex (path-replace-extension pathname ".hex")] ; path to .hex file
                                   [path-dir (path-only pathname)] ; directory of the source file (for .include directives of the assembler (-I command-line option))
                                   [as-path (find-executable-path (send pref-dialog get-avr-as))] ; path to avr-as
                                   [ld-path (find-executable-path (send pref-dialog get-avr-ld))] ; path to avr-ld
                                   [objcopy-path (find-executable-path (send pref-dialog get-avr-objcopy))] ; path to avr-objcopy
                                   [size-path (find-executable-path (send pref-dialog get-avr-size))]) ; path to avr-size
                              (if (and as-path ld-path objcopy-path size-path) ; build only if all paths of the toolchain are correct
                                  (let ([commands ; chain of commands
                                         (list ; list of pairs (tool . (path-to-exe . arg-list))
                                          (cons 'as (cons as-path (avr-as-parameters (send pref-dialog get-avr-as-isa-mcu-string) path-dir path-o path-s)))
                                          (cons 'ld (cons ld-path (avr-ld-parameters path-elf path-o)))
                                          (cons 'objcopy (cons objcopy-path (avr-objcopy-parameters path-elf path-hex)))
                                          (cons 'size (cons size-path (avr-size-parameters path-elf))))])
                                    (for-each (lambda (command)
                                                ; launch process and loop to get its output
                                                (let ([path-string (cadr command)] [arg-list (cddr command)])
                                                  (let-values ([(process-id port-stdout port-stdin port-stderr) (apply subprocess #f #f #f path-string arg-list)])
                                                    (if (eq? (car command) 'size)
                                                        ; avr-size (specially processed to analyse its output)
                                                        (begin
                                                          (subprocess-wait process-id) ; wait for 'avr-size' to terminate
                                                          (let ([size-result (string-trim (port->string port-stdout))]) ; 'port->string' closes the port
                                                            (if (non-empty-string? size-result) ; empty in case of build error ('size' thus fails)
                                                                (let* ([total-line (last (string-split size-result "\n"))] ; get last line, then the value only
                                                                       [bytes (string->number (last (string-split total-line)))]
                                                                       [bytes-str (string-append "Machine code: "
                                                                                                 (if (< bytes 1024) (format "~a bytes (" bytes) (format "~a KiB (" (ceiling (/ bytes 1024))))
                                                                                                 source-file-name ")")])
                                                                  (send build-message set-label bytes-str))
                                                                (send build-message set-label ""))) ; remove message, no need to show 'building error'
                                                          (close-output-port port-stdin)
                                                          (close-input-port port-stderr))
                                                        ; otherwise (avr-as, avr-ld, avr-objcopy)
                                                        ; outer loop: read port while the process is running
                                                        (do ([port-outerr (merge-input port-stdout port-stderr)]) ; id init-expr step-expr
                                                          ((not (eq? (subprocess-status process-id) 'running)) ; stop-expr
                                                           ; finish-exprs
                                                           (close-input-port port-outerr) ; doc: "Closing the merged ports does not close the original ports."
                                                           (close-input-port port-stdout)
                                                           (close-output-port port-stdin)
                                                           (close-input-port port-stderr))
                                                          ; inner loop: loop while there is something to read in the port
                                                          (do ([line-outerr (read-line port-outerr 'any) (read-line port-outerr 'any)])
                                                            ((eof-object? line-outerr) ; stop-expr
                                                             ; finish-exprs
                                                             )
                                                            (send output-editor insert line-outerr)
                                                            (send output-editor insert #\newline)))
                                                        ))))
                                              commands))
                                  ; otherwise (i.e. some paths are incorrect)
                                  (send output-editor insert " ! Building has failed: some paths of the toolchain are incorrect.\n")
                                  )
                              (send output-editor insert "\n--- building has completed ---\n")

                              (when upload-after-build? ; Now that building has been completed, upload if needed
                                (send output-editor insert "--- uploading has started ---\n")
                                (let* ([path-string (find-executable-path (send pref-dialog get-avrdude))]
                                       [avrdude-conf-path (find-executable-path (send pref-dialog get-avrdude-conf))]
                                       [arg-list (avrdude-parameters avrdude-conf-path (car (list-ref avrdude-parts (send pref-dialog get-avrdude-part)))
                                                                     (send pref-dialog get-avrdude-programmer-string) (send pref-dialog get-port) path-hex)])
                                  (if (and path-string avrdude-conf-path)
                                      (let-values ([(process-id port-stdout port-stdin port-stderr)
                                                    (apply subprocess #f #f #f path-string arg-list)])
                                        (set! avrdude-process-id process-id)
                                        (send kill-item enable #t)
                                        (do ([port-outerr (merge-input port-stdout port-stderr)]) ; id init-expr step-expr
                                          ((not (eq? (subprocess-status process-id) 'running)) ; stop-expr
                                           ; finish-exprs
                                           (close-input-port port-outerr) ; doc: "Closing the merged ports does not close the original ports."
                                           (close-input-port port-stdout)
                                           (close-output-port port-stdin)
                                           (close-input-port port-stderr))
                                          ; inner loop: loop while there is something to read in the port
                                          (do ([line-outerr (read-line port-outerr 'any) (read-line port-outerr 'any)]) ; id init-expr step-expr
                                            ((eof-object? line-outerr) ; stop-expr
                                             ; finish-exprs
                                             )
                                            (send output-editor insert line-outerr)
                                            (send output-editor insert #\newline)))
                                        (send kill-item enable #f)
                                        (set! avrdude-process-id #f))
                                      ; otherwise (i.e. the path of avrdude or avrdude.conf is wrong)
                                      (send output-editor insert " ! Uploading has failed: the path of avrdude or avrdude.conf is incorrect.\n")
                                      ))
                                (send output-editor insert "\n--- uploading has completed ---\n"))
                              )
                            
                            (send output-editor lock #t)
                            (end-busy-cursor)
                            (for-each (lambda (control) (send control enable #t)) to-disable)
                            )
                          ))))))
    
    (define avrdude-process-id #f) ; used to kill avrdude if requested by the user
    (define (callback-kill)
      (when avrdude-process-id
        (subprocess-kill avrdude-process-id #t)
        (set! avrdude-process-id #f)))
    
    ; - child dialogs -
    ; Find
    (define search-frame (new frame% [label "Find and replace"] [parent this] [min-width 250] [border 2] [style '(no-resize-border)] [stretchable-width #f] [stretchable-height #f]))
    (define search-edit (new text-field% [label "Search:"] [parent search-frame]))
    (define replace-edit (new text-field% [label "Replace with:"] [parent search-frame]))
    (define search-group (new group-box-panel% [label "Parameters"] [parent search-frame] [alignment '(left top)]))
    (define direction-radio (new radio-box% [label "Direction:"] [parent search-group] [choices '("Forward" "Backward")] [style '(horizontal)]))
    (define case-checkbox (new check-box% [label "Case-sensitive"] [parent search-group]))
    (define search-horizontal1 (new horizontal-pane% [parent search-frame] [alignment '(center center)]))
    (define search-button (new button% [label "Find"] [parent search-horizontal1] [style '(border)]
                               [callback (lambda (button event) (send search-frame show #f)
                                           (callback-find (send search-edit get-value) (if (= (send direction-radio get-selection) 0) 'forward 'backward) (send case-checkbox get-value)))]))
    (define replace-button (new button% [label "Replace"] [parent search-horizontal1]
                                [callback (lambda (button event) (send search-frame show #f)
                                            (callback-replace (send search-edit get-value) (send replace-edit get-value) (if (= (send direction-radio get-selection) 0) 'forward 'backward) (send case-checkbox get-value)))]))

    ; Preferences
    (define pref-dialog (new preferences-dialog% [parent this]))

    ; View settings
    (define view-dialog (new view-dialog% [parent this]))
    
    ; About
    (define about-dialog (new about-dialog% [parent this]))
    
    ; - window construction -
    ; Main menu
    (define main-menu (new menu-bar% [parent this]))
    (define file-menu (new menu% [label "&File"] [parent main-menu]))
    (new menu-item% [parent file-menu] [label "&New tab"] [shortcut #\t] [callback (lambda (menu-item event) (send tab-panel on-new-request))])
    (new menu-item% [parent file-menu] [label "&Open..."] [shortcut #\o] [callback (lambda (menu-item event) (on-drop-file ""))])
    (define recent-menu (new menu% [parent file-menu] [label "Open recen&t"]))
    (new menu-item% [parent file-menu] [label "&Save"] [shortcut #\s] [callback (lambda (menu-item event) (send (send tab-panel get-current-editor) save-file #f 'text))])
    (new menu-item% [parent file-menu] [label "Save &as..."] [callback (lambda (menu-item event) (send (send tab-panel get-current-editor) save-file "" 'text))])
    (new menu-item% [parent file-menu] [label "&Reload"] [callback (lambda (menu-item event) (let ([current-editor (send tab-panel get-current-editor)]) (when (send current-editor can-close?) (send current-editor load-file #f 'text))))])
    (define menu-close-item (new menu-item% [parent file-menu] [label "&Close tab..."] [shortcut #\w] [callback (lambda (menu-item event) (send tab-panel on-close-request (send tab-panel get-selection)))]))
    (new separator-menu-item% [parent file-menu])
    (new menu-item% [parent file-menu] [label "&Preferences..."] [callback (lambda (menu-item event) (send pref-dialog show #t))])
    (new separator-menu-item% [parent file-menu])
    (new menu-item% [parent file-menu] [label "&Exit"] [shortcut #\q] [callback (lambda (menu-item event) (when (send this can-exit?) (send this on-exit)))])
    (define edit-menu (new menu% [parent main-menu] [label "&Edit"]))
    (append-editor-operation-menu-items edit-menu)
    (new separator-menu-item% [parent edit-menu])
    (new menu-item% [label "&Find and replace..."] [parent edit-menu] [shortcut #\f] [callback (lambda (menu-item event) (send search-frame center) (send search-edit focus) (send search-frame show #t))])
    (new menu-item% [label "Find &next"] [parent edit-menu] [shortcut #\g] [callback (lambda (menu-item event) (callback-find (send search-edit get-value) 'forward (send case-checkbox get-value)))])
    (new menu-item% [label "Find &previous"] [parent edit-menu] [shortcut-prefix '(ctl shift)] [shortcut #\g] [callback (lambda (menu-item event) (callback-find (send search-edit get-value) 'backward (send case-checkbox get-value)))])
    (new menu-item% [label "Replace next"] [parent edit-menu] [shortcut #\r] [callback (lambda (menu-item event) (callback-replace (send search-edit get-value) (send replace-edit get-value) 'forward (send case-checkbox get-value)))])
    (new menu-item% [label "Replace previous"] [parent edit-menu] [shortcut-prefix '(ctl shift)] [shortcut #\r] [callback (lambda (menu-item event) (callback-replace (send search-edit get-value) (send replace-edit get-value) 'backward (send case-checkbox get-value)))])
    (define view-menu (new menu% [parent main-menu] [label "&View"]))
    (define linewrap-item (new checkable-menu-item% [parent view-menu] [label "Line wrap"]
                               [callback (lambda (menu-item event) (for ([editor (send tab-panel get-all-editors)]) (send editor auto-wrap (send menu-item is-checked?))))]
                               [checked (get-preference 'linewrap (lambda () default-linewrap) 'timestamp preferences-filename)]))
    (new menu-item% [label "&Toggle output"] [parent view-menu] [shortcut #\e] [callback (lambda (menu-item event) (callback-toggle))])
    (new separator-menu-item% [parent view-menu])
    (new menu-item% [parent view-menu] [label "&Settings..."] [callback (lambda (menu-item event) (send view-dialog show #t))])
    (new separator-menu-item% [parent view-menu])
    (define reset-item (new checkable-menu-item% [label "&Reset on restart"] [parent view-menu] [callback (lambda (menu-item event) (void))]))
    (define tools-menu (new menu% [parent main-menu] [label "&Tools"]))
    (define build-item (new menu-item% [label "&Build"] [parent tools-menu] [shortcut #\b] [callback (lambda (menu-item event) (send build-button command event))]))
    (define upload-item (new menu-item% [label "&Upload"] [parent tools-menu] [shortcut #\u] [callback (lambda (menu-item event) (send upload-button command event))]))
    (define kill-item (new menu-item% [label "&Stop upload"] [parent tools-menu] [callback (lambda (menu-item event) (callback-kill))]))
    (new separator-menu-item% [parent tools-menu])
    (new menu-item% [parent tools-menu] [label "&Explore folder"] [callback (lambda (menu-item event) (callback-explorer))])
    (define help-menu (new menu% [parent main-menu] [label "&Help"]))
    (new menu-item% [label "&Homepage"] [parent help-menu] [callback (lambda (menu-item event) (callback-homepage))])
    (new menu-item% [label "&Shortcuts..."] [parent help-menu] [shortcut 'f2] [shortcut-prefix '()] [callback (lambda (menu-item event)
                                                                                                                (message-box "Shortcuts"
                                                                                                                             "Non-trivial keyboard shortcuts available in A-IDE:
Ctrl-E\tTo show or hide the output pane.
Ctrl-B\tTo build the edited assembly code.
Ctrl-U\tTo upload the machine code to the microcontroller.
Ctrl-K\tTo delete the rest of the current line.
Ctrl-Del\tTo delete the current line.
Ctrl-Alt-O\tToggles the insertion/overwrite edition mode.
Alt-G\tGo to a given line number."
                                                                                                                             this '(ok no-icon))
                                                                                                                )])
    (new menu-item% [label "&About..."] [parent help-menu] [shortcut 'f1] [shortcut-prefix '()] [callback (lambda (menu-item event) (callback-about))])
    
    ; Layout
    (define v0-layout (new panel:vertical-dragable% [parent this])) ; upper part (toolbar, tabs) + output
    (define v1-layout (new vertical-pane% [parent v0-layout])) ; toolbar + tabs (so as to always stick them, not adjustable)
    
    ; Toolbar
    (define h0-layout (new horizontal-pane% [parent v1-layout] [stretchable-height #f]))
    (new message% [label "Line"] [parent h0-layout] [horiz-margin 8])
    (define line-label (new line-message% [parent h0-layout] [horiz-margin 0]))
    (define h1-layout (new horizontal-pane% [parent h0-layout] [alignment '(center center)]))
    (define build-message (new message% [label ""] [parent h1-layout] [horiz-margin 10] [auto-resize #t]))
    (define h2-layout (new horizontal-pane% [parent h0-layout] [alignment '(right center)] [stretchable-width #f]))
    (define build-button (new button% [parent h2-layout] [label "&Build"] [callback (lambda (button event) (callback-build #f))]))
    (define upload-button (new button% [parent h2-layout] [label "&Upload"] [callback (lambda (button event) (callback-build #t))]))
    
    ; Editor pane (tabs)
    (define tab-panel (new ide-tab-panel% [parent v1-layout] [frame this]))

    ; Output pane
    (define output-editor (new text%))
    (define output-canvas (new editor-canvas% [parent v0-layout] [editor output-editor] [style '(auto-hscroll)]))
    
    ; - window initialisation -
    ; layout settings
    (if (get-preference 'output-shown (lambda () default-output-shown) 'timestamp preferences-filename)
        (send v0-layout set-percentages percentages) ; the output pane stays visible: set percentages
        (send v0-layout delete-child output-canvas)) ; hide the output pane
    
    ; menu settings
    (send menu-close-item enable #f)
    (send kill-item enable #f)
    (populate-recent-menu)

    ; output editor settings (code editor settings are done directly in the ide-tab-panel.rkt source file)
    (define output-keymap (new keymap%))
    (add-editor-keymap-functions output-keymap) ; safer
    (add-text-keymap-functions output-keymap) ; otherwise keymap:setup-global triggers errors when some keys are pressed
    (keymap:setup-global output-keymap) ; especially to add popup menu to output pane
    ;(keymap:setup-editor output-keymap)
    (send (send output-editor get-keymap) chain-to-keymap output-keymap #f)
    (send output-canvas set-canvas-background (send view-dialog get-output-background))
    (define output-editor-style ; because calling 'erase' removes styling, we store the style for later re-application
      (let* ([style-list (send output-editor get-style-list)] [original-style (send (send (send tab-panel get-current-editor) get-style-list) find-named-style "base-font")])
        (send style-list convert original-style)))
    (send output-editor change-style output-editor-style) ; /!\ the editor style-list must contain the style!
    (send output-editor auto-wrap #t)
    (send output-editor lock #t) ; lock output editor (the user is not supposed to input content there)
    
    ; misc settings
    (send this accept-drop-files #t)
    ))
